<?php
// user.php
require_once 'config.php';
require_once 'db_connection.php';

// ইউজার অথেন্টিকেশন
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user = $db->select('users', '*', ['id' => $user_id])[0];

// যদি ইউজার ব্লক করা থাকে
if ($user['status'] != 'active') {
    session_destroy();
    header('Location: index.php?error=blocked');
    exit();
}

// ব্যালেন্স
$deposit_balance = $user['deposit_balance'];
$winning_balance = $user['winning_balance'];
$total_balance = $deposit_balance + $winning_balance;

// ট্রানজেকশন হিস্টরি
$transactions = $db->query("
    (SELECT 'Deposit' as type, amount, status, created_at, '' as details FROM deposits WHERE user_id = ?)
    UNION
    (SELECT 'Withdrawal' as type, amount, status, created_at, CONCAT(method, ' - ', account_number) as details FROM withdrawals WHERE user_id = ?)
    UNION
    (SELECT 'Bet' as type, amount, CASE WHEN outcome = 'Win' THEN 'approved' ELSE 'pending' END as status, created_at, bet_type as details FROM bets WHERE user_id = ?)
    ORDER BY created_at DESC
    LIMIT 20
", [$user_id, $user_id, $user_id])->fetch_all(MYSQLI_ASSOC);

// বেট হিস্টরি
$bet_history = $db->select('bets', '*', ['user_id' => $user_id], 'created_at DESC LIMIT 20');

// গেম হিস্টরি (API থেকে)
$game_history = [];
try {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, HISTORY_API);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);
    
    if ($response) {
        $data = json_decode($response, true);
        if (isset($data['data']['list'])) {
            $game_history = array_slice($data['data']['list'], 0, 10);
        }
    }
} catch (Exception $e) {
    // API error - use dummy data
    $game_history = [];
}

// পেমেন্ট নাম্বার
$payment_numbers = $db->select('payment_numbers', '*', ['is_active' => 1]);

// সেটিংস
$settings_result = $db->select('settings', '*');
$settings = [];
foreach ($settings_result as $setting) {
    $settings[$setting['setting_key']] = $setting['setting_value'];
}

// AJAX requests
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'place_bet':
            $bet_type = $db->sanitize($_POST['bet_type']);
            $amount = floatval($_POST['amount']);
            $period = $_POST['period'];
            
            // ব্যালেন্স চেক
            if ($total_balance < $amount) {
                echo json_encode(['success' => false, 'message' => 'Insufficient balance']);
                exit();
            }
            
            $db->beginTransaction();
            try {
                // ব্যালেন্স কাট
                if ($deposit_balance >= $amount) {
                    $new_deposit = $deposit_balance - $amount;
                    $db->update('users', ['deposit_balance' => $new_deposit], ['id' => $user_id]);
                } else {
                    $remaining = $amount - $deposit_balance;
                    $new_deposit = 0;
                    $new_winning = $winning_balance - $remaining;
                    $db->update('users', [
                        'deposit_balance' => $new_deposit,
                        'winning_balance' => $new_winning
                    ], ['id' => $user_id]);
                }
                
                // বেট রেকর্ড
                $bet_id = uniqid('bet_');
                $db->insert('bets', [
                    'bet_id' => $bet_id,
                    'user_id' => $user_id,
                    'bet_type' => $bet_type,
                    'amount' => $amount,
                    'period' => $period,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
                
                $db->commit();
                echo json_encode(['success' => true, 'message' => 'Bet placed!']);
            } catch (Exception $e) {
                $db->rollback();
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit();
            
        case 'deposit_request':
            $amount = floatval($_POST['amount']);
            $method = $db->sanitize($_POST['method']);
            $agent_number = $db->sanitize($_POST['agent_number']);
            $transaction_id = $db->sanitize($_POST['transaction_id']);
            
            if ($amount < ($settings['min_deposit'] ?? MIN_DEPOSIT)) {
                echo json_encode(['success' => false, 'message' => 'Minimum deposit is ৳' . ($settings['min_deposit'] ?? MIN_DEPOSIT)]);
                exit();
            }
            
            $deposit_id = uniqid('dep_');
            $result = $db->insert('deposits', [
                'deposit_id' => $deposit_id,
                'user_id' => $user_id,
                'amount' => $amount,
                'method' => $method,
                'agent_number' => $agent_number,
                'transaction_id' => $transaction_id,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            if (isset($result['error'])) {
                echo json_encode(['success' => false, 'message' => $result['error']]);
            } else {
                echo json_encode(['success' => true, 'message' => 'Deposit request submitted!']);
            }
            exit();
            
        case 'withdraw_request':
            $amount = floatval($_POST['amount']);
            $method = $db->sanitize($_POST['method']);
            $account_number = $db->sanitize($_POST['account_number']);
            
            if ($amount < ($settings['min_withdraw'] ?? MIN_WITHDRAW)) {
                echo json_encode(['success' => false, 'message' => 'Minimum withdrawal is ৳' . ($settings['min_withdraw'] ?? MIN_WITHDRAW)]);
                exit();
            }
            
            if ($winning_balance < $amount) {
                echo json_encode(['success' => false, 'message' => 'Insufficient winning balance']);
                exit();
            }
            
            $withdrawal_id = uniqid('wd_');
            $result = $db->insert('withdrawals', [
                'withdrawal_id' => $withdrawal_id,
                'user_id' => $user_id,
                'amount' => $amount,
                'method' => $method,
                'account_number' => $account_number,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            if (isset($result['error'])) {
                echo json_encode(['success' => false, 'message' => $result['error']]);
            } else {
                // Provisional balance deduction
                $db->update('users', [
                    'winning_balance' => $winning_balance - $amount
                ], ['id' => $user_id]);
                
                echo json_encode(['success' => true, 'message' => 'Withdrawal request submitted!']);
            }
            exit();
            
        case 'redeem_gift_code':
            $code = $db->sanitize($_POST['code']);
            
            // কোড চেক
            $gift_code = $db->select('gift_codes', '*', ['code' => $code, 'status' => 'active']);
            
            if (!$gift_code) {
                echo json_encode(['success' => false, 'message' => 'Invalid or expired code']);
                exit();
            }
            
            $gift_code = $gift_code[0];
            
            // রিডিম
            $db->beginTransaction();
            try {
                $db->update('gift_codes', [
                    'status' => 'redeemed',
                    'redeemed_by' => $user_id,
                    'redeemed_at' => date('Y-m-d H:i:s')
                ], ['id' => $gift_code['id']]);
                
                $db->query("UPDATE users SET deposit_balance = deposit_balance + ? WHERE id = ?", 
                    [$gift_code['amount'], $user_id]);
                
                $db->commit();
                echo json_encode(['success' => true, 'message' => 'Redeemed ৳' . $gift_code['amount'] . '!']);
            } catch (Exception $e) {
                $db->rollback();
                echo json_encode(['success' => false, 'message' => 'Redemption failed']);
            }
            exit();
    }
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - User Dashboard</title>
    <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@400;700&family=Roboto+Mono:wght@700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="styles.css">
    <style>
        /* Your original user CSS styles from HTML */
        :root {
            --primary-color: #00ffc8;
            --secondary-color: #00aaff;
            --accent-red: #ff3366;
            --accent-yellow: #FFD700;
            --dark-bg: #0e0e0e;
            --card-bg: #1f1f1f;
            --header-bg: #2c2c2e;
            --text-primary: #ffffff;
            --text-secondary: #a9a9b3;
            --gradient-primary: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            --gradient-danger: linear-gradient(135deg, #ff3366, #ff6b8b);
            --gradient-success: linear-gradient(135deg, #4CAF50, #81C784);
        }
        /* ... All your user CSS styles ... */
    </style>
</head>
<body>
    <!-- SPLASH SCREEN (hidden after load) -->
    <div id="splash-screen" style="display: none;">
        <div class="splash-loader"></div>
        <div class="splash-image-container">
            <?php include 'splash_config.php'; ?>
            <img src="<?php echo SPLASH_IMAGE_URL; ?>" alt="Welcome">
        </div>
    </div>

    <!-- MAIN CONTAINER -->
    <div class="container">
        <header class="header">
            <h1><?php echo APP_NAME; ?></h1>
            <div class="balance-section">
                <span>Balance: ৳</span>
                <span id="user-balance"><?php echo number_format($total_balance, 2); ?></span>
                <button id="refresh-balance-btn" title="Refresh Balance" onclick="refreshBalance()">
                    <i class="fas fa-sync-alt"></i>
                </button>
            </div>
        </header>

        <!-- NOTICE -->
        <div class="notice-bar">
            <div class="notice-content">
                <?php echo htmlspecialchars($settings['notice_text'] ?? 'স্বাগতম! 💖 সর্বনিম্ন ৫০ টাকা ডিপোজিট করুন। প্রথমবার ৮০ টাকা ডিপোজিট করলেই ২০ টাকা বোনাস! 🎁'); ?>
            </div>
        </div>

        <!-- NAVIGATION -->
        <div class="nav-tabs">
            <button class="nav-tab active" data-page="home">Home</button>
            <button class="nav-tab" data-page="wallet">Wallet</button>
            <button class="nav-tab" data-page="profile">Profile</button>
        </div>

        <!-- HOME PAGE -->
        <div id="home-page" class="page-content active">
            <!-- GAME CONTAINER -->
            <main class="game-container">
                <div id="timer-overlay">
                    <span id="overlay-countdown-text"></span>
                </div>

                <section class="timer-section">
                    <div class="timer-info">
                        <div>
                            <h2>WinGo 1 Min</h2>
                            <p>Period: <span id="currentPeriod"><?php echo time(); ?></span></p>
                        </div>
                        <div>
                            <p>Time Remaining</p>
                            <div id="countdown">00:00</div>
                        </div>
                    </div>
                </section>

                <div class="betting-area">
                    <section class="betting-options">
                        <button class="btn btn-green" data-bet-type="Green">Green</button>
                        <button class="btn btn-violet" data-bet-type="Violet">Violet</button>
                        <button class="btn btn-red" data-bet-type="Red">Red</button>
                    </section>
                    <section class="number-grid">
                        <button class="btn" data-bet-type="0">0</button>
                        <button class="btn btn-num-green" data-bet-type="1">1</button>
                        <button class="btn btn-num-red" data-bet-type="2">2</button>
                        <button class="btn btn-num-green" data-bet-type="3">3</button>
                        <button class="btn btn-num-red" data-bet-type="4">4</button>
                        <button class="btn" data-bet-type="5">5</button>
                        <button class="btn btn-num-red" data-bet-type="6">6</button>
                        <button class="btn btn-num-green" data-bet-type="7">7</button>
                        <button class="btn btn-num-red" data-bet-type="8">8</button>
                        <button class="btn btn-num-green" data-bet-type="9">9</button>
                    </section>
                    <section class="size-options">
                        <button class="btn btn-big" data-bet-type="Big">Big</button>
                        <button class="btn btn-small" data-bet-type="Small">Small</button>
                    </section>
                </div>
            </main>

            <!-- HISTORY TABS -->
            <div class="tabs">
                <button class="tab-btn active" data-tab="game-history">Game History</button>
                <button class="tab-btn" data-tab="my-history">My History</button>
            </div>

            <div id="game-history-panel" class="tab-panel active">
                <div class="history-container">
                    <table>
                        <thead>
                            <tr><th>Period</th><th>Number</th><th>Big/Small</th><th>Color</th></tr>
                        </thead>
                        <tbody id="resultBody">
                            <?php if (!empty($game_history)): ?>
                                <?php foreach ($game_history as $game): ?>
                                <tr>
                                    <td><?php echo $game['issueNumber'] ?? ''; ?></td>
                                    <td><?php echo $game['number'] ?? ''; ?></td>
                                    <td><?php echo isset($game['number']) && $game['number'] >= 5 ? 'Big' : 'Small'; ?></td>
                                    <td>
                                        <?php 
                                        $num = $game['number'] ?? 0;
                                        if (in_array($num, [1,3,7,9])) echo '<span class="dot green"></span>';
                                        elseif (in_array($num, [2,4,6,8])) echo '<span class="dot red"></span>';
                                        elseif ($num == 5) echo '<span class="dot green"></span> <span class="dot violet"></span>';
                                        elseif ($num == 0) echo '<span class="dot red"></span> <span class="dot violet"></span>';
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr><td colspan="4">Loading history...</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="my-history-panel" class="tab-panel">
                <div class="history-container">
                    <table id="my-history-table">
                        <thead>
                            <tr><th>Period</th><th>Bet</th><th>Amount</th><th>Result</th></tr>
                        </thead>
                        <tbody id="myHistoryBody">
                            <?php foreach ($bet_history as $bet): ?>
                            <tr>
                                <td><?php echo $bet['period']; ?></td>
                                <td><?php echo htmlspecialchars($bet['bet_type']); ?></td>
                                <td>৳<?php echo number_format($bet['amount'], 2); ?></td>
                                <td class="<?php echo $bet['outcome'] == 'Win' ? 'outcome-win' : 'outcome-loss'; ?>">
                                    <?php 
                                    if ($bet['outcome'] == 'Pending') echo 'Pending...';
                                    elseif ($bet['outcome'] == 'Win') echo '+৳' . number_format($bet['winnings'], 2);
                                    else echo '-৳' . number_format($bet['amount'], 2);
                                    ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- WALLET PAGE -->
        <div id="wallet-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('home')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Wallet</h2>
            </div>
            
            <div class="wallet-card">
                <div class="wallet-balance">
                    <h2>Available Balance</h2>
                    <p>৳<span id="wallet-balance-total"><?php echo number_format($total_balance, 2); ?></span></p>
                    <div class="balance-breakdown">
                        <div>
                            <h3>Deposit</h3>
                            <p>৳<span id="wallet-balance-deposit"><?php echo number_format($deposit_balance, 2); ?></span></p>
                        </div>
                        <div>
                            <h3>Winning</h3>
                            <p>৳<span id="wallet-balance-winning"><?php echo number_format($winning_balance, 2); ?></span></p>
                        </div>
                    </div>
                </div>
                <div class="wallet-actions">
                    <button class="wallet-action-btn deposit-btn" onclick="navigateToPage('deposit-select')">
                        <i class="fas fa-arrow-down"></i> Deposit
                    </button>
                    <button class="wallet-action-btn withdraw-btn" id="withdraw-nav-btn" onclick="navigateToPage('withdraw')">
                        <i class="fas fa-arrow-up"></i> Withdraw
                    </button>
                </div>
            </div>

            <div class="transaction-history">
                <h3>Transaction History</h3>
                <div class="transaction-history-list" id="transaction-list">
                    <?php foreach ($transactions as $txn): ?>
                    <div class="transaction-item">
                        <div class="transaction-info">
                            <div class="transaction-type"><?php echo $txn['type']; ?> (<?php echo $txn['status']; ?>)</div>
                            <div class="transaction-date"><?php echo date('d/m/Y H:i', strtotime($txn['created_at'])); ?></div>
                            <?php if ($txn['details']): ?>
                            <div class="transaction-details"><?php echo htmlspecialchars($txn['details']); ?></div>
                            <?php endif; ?>
                        </div>
                        <div class="transaction-amount <?php echo $txn['type'] == 'Deposit' ? 'credit' : 'debit'; ?>">
                            <?php echo $txn['type'] == 'Deposit' ? '+' : '-'; ?>৳<?php echo number_format($txn['amount'], 2); ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- DEPOSIT PAGE -->
        <div id="deposit-select-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('wallet')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Deposit</h2>
            </div>
            
            <div class="payment-tabs">
                <div class="payment-tab active" onclick="switchPaymentMethod('bKash')">bKash</div>
                <div class="payment-tab" onclick="switchPaymentMethod('Nagad')">Nagad</div>
            </div>
            
            <div class="deposit-notice">
                এই অ্যাপে ডিপোজিট করার জন্য অবশ্যই <strong>সেন্ড মানি</strong> করতে হবে।
            </div>
            
            <div id="payment-number-container">
                <?php foreach ($payment_numbers as $number): ?>
                <div class="number-card <?php echo $number['type'] == 'bKash' ? 'selected' : ''; ?>" data-type="<?php echo $number['type']; ?>">
                    <div class="label"><?php echo htmlspecialchars($number['label']); ?></div>
                    <div class="number"><?php echo htmlspecialchars($number['number']); ?></div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="form-group">
                <label for="deposit-amount-input">টাকার পরিমাণ</label>
                <input type="number" id="deposit-amount-input" placeholder="Enter amount" min="<?php echo $settings['min_deposit'] ?? MIN_DEPOSIT; ?>">
                <div class="amount-quick-select">
                    <button class="amount-btn" onclick="setAmount(100)">৳100</button>
                    <button class="amount-btn" onclick="setAmount(500)">৳500</button>
                    <button class="amount-btn" onclick="setAmount(1000)">৳1000</button>
                    <button class="amount-btn" onclick="setAmount(5000)">৳5000</button>
                </div>
            </div>
            
            <button class="submit-btn" onclick="submitDeposit()">Next</button>
        </div>

        <!-- WITHDRAW PAGE -->
        <div id="withdraw-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('wallet')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Withdraw</h2>
            </div>
            
            <div class="wallet-card">
                <div class="wallet-balance">
                    <h2>Available for Withdrawal</h2>
                    <p>৳<span id="withdrawal-available-balance"><?php echo number_format($winning_balance, 2); ?></span></p>
                </div>
                
                <div class="form-group">
                    <label for="withdraw-amount">Amount (৳)</label>
                    <input type="number" id="withdraw-amount" placeholder="Amount" min="<?php echo $settings['min_withdraw'] ?? MIN_WITHDRAW; ?>">
                </div>
                
                <div class="form-group">
                    <label for="withdraw-method">Payment Method</label>
                    <select id="withdraw-method">
                        <option value="bKash">bKash</option>
                        <option value="Nagad">Nagad</option>
                        <option value="Rocket">Rocket</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="withdraw-account">Account Number</label>
                    <input type="text" id="withdraw-account" placeholder="Your account number">
                </div>
                
                <button class="submit-btn" onclick="submitWithdrawal()">Submit Request</button>
            </div>
        </div>

        <!-- PROFILE PAGE -->
        <div id="profile-page" class="page-content">
            <div class="profile-card">
                <div class="profile-avatar">
                    <i class="fas fa-user"></i>
                </div>
                <div class="profile-info">
                    <h2 id="user-name"><?php echo htmlspecialchars($user['name']); ?></h2>
                    <p>UID: <span id="user-id"><?php echo htmlspecialchars($user['ui_id']); ?></span></p>
                    <p>Phone: <span id="user-phone"><?php echo htmlspecialchars($user['phone']); ?></span></p>
                    <p>Referral Code: <span id="referral-code"><?php echo htmlspecialchars($user['ui_id']); ?></span></p>
                </div>
            </div>
            
            <div class="profile-menu-list">
                <div class="profile-menu-item" onclick="navigateToPage('wallet')">
                    <i class="fas fa-wallet"></i>
                    <span>Wallet</span>
                </div>
                <div class="profile-menu-item" onclick="navigateToPage('referral')">
                    <i class="fas fa-user-friends"></i>
                    <span>Referral</span>
                </div>
                <div class="profile-menu-item" onclick="navigateToPage('bonus')">
                    <i class="fas fa-gift"></i>
                    <span>Bonus</span>
                </div>
                <div class="profile-menu-item" onclick="navigateToPage('settings')">
                    <i class="fas fa-cog"></i>
                    <span>Settings</span>
                </div>
            </div>
            
            <button class="submit-btn" onclick="logout()" style="background: var(--gradient-danger); margin-top: 20px;">
                <i class="fas fa-sign-out-alt"></i> Logout
            </button>
        </div>

        <!-- REFERRAL PAGE -->
        <div id="referral-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('profile')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Referral</h2>
            </div>
            
            <div class="referral-card">
                <h2>Your Referral Code</h2>
                <div class="referral-code-container">
                    <div class="referral-code" id="referral-code-display"><?php echo htmlspecialchars($user['ui_id']); ?></div>
                    <button class="copy-btn" onclick="copyReferralCode()">
                        <i class="fas fa-copy"></i> Copy
                    </button>
                </div>
                <p>Share this code with your friends and earn bonus when they register!</p>
            </div>

            <div class="referral-stats">
                <div class="stat-card">
                    <div class="stat-value" id="referral-count"><?php echo $user['total_referrals']; ?></div>
                    <div class="stat-label">Total Referrals</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="referral-bonus">৳<?php echo number_format($user['total_bonus'], 2); ?></div>
                    <div class="stat-label">Total Bonus</div>
                </div>
            </div>
        </div>

        <!-- BONUS PAGE -->
        <div id="bonus-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('profile')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Bonus</h2>
            </div>
            
            <div class="gift-code-card">
                <h2>Redeem Gift Code</h2>
                <div class="gift-code-form">
                    <div class="form-group">
                        <label for="gift-code-input-field">Enter Gift Code</label>
                        <input type="text" id="gift-code-input-field" placeholder="Enter your gift code here">
                    </div>
                    <button class="submit-btn" onclick="redeemGiftCode()">Redeem</button>
                </div>
            </div>
        </div>

        <!-- SETTINGS PAGE -->
        <div id="settings-page" class="page-content">
            <div class="page-header">
                <button class="back-btn" onclick="navigateToPage('profile')"><i class="fas fa-arrow-left"></i></button>
                <h2 class="page-title">Settings</h2>
            </div>
            
            <div class="profile-card">
                <h2>Change Password</h2>
                <div class="auth-form">
                    <div class="form-group">
                        <label for="current-password">Current Password</label>
                        <input type="password" id="current-password" placeholder="Current Password">
                    </div>
                    <div class="form-group">
                        <label for="new-password">New Password</label>
                        <input type="password" id="new-password" placeholder="New Password">
                    </div>
                    <div class="form-group">
                        <label for="confirm-new-password">Confirm New Password</label>
                        <input type="password" id="confirm-new-password" placeholder="Confirm New Password">
                    </div>
                    <button class="submit-btn" onclick="changePassword()">Update Password</button>
                </div>
            </div>
        </div>
    </div>

    <!-- POPUPS -->
    <div class="popup-overlay" id="bet-popup">
        <div class="popup-content bet-popup-content">
            <h3 id="bet-popup-title">Place Bet on <span>...</span></h3>
            <div class="bet-popup-balance">
                Available: <span>৳</span><span id="bet-popup-current-balance"><?php echo number_format($total_balance, 2); ?></span>
            </div>
            <p>আপনার বেটের পরিমাণ লিখুন:</p>
            <div class="bet-input-container">
                <span>৳</span>
                <input type="number" id="bet-amount" placeholder="Amount" min="1">
            </div>
            <div class="bet-quick-select">
                <button onclick="addBetAmount(10)">+10</button>
                <button onclick="addBetAmount(50)">+50</button>
                <button onclick="addBetAmount(100)">+100</button>
                <button onclick="multiplyBetAmount(2)">x2</button>
            </div>
            <div class="popup-buttons">
                <button onclick="closeBetPopup()">Cancel</button>
                <button onclick="confirmBet()" style="background: var(--gradient-primary); color: var(--dark-bg);">Confirm</button>
            </div>
        </div>
    </div>

    <div class="popup-overlay" id="result-popup">
        <div class="result-popup-content" id="result-popup-content">
            <!-- Dynamic content -->
        </div>
    </div>

    <div class="popup-overlay" id="alert-popup">
        <div class="popup-content" id="alert-popup-content">
            <h3 id="alert-title"></h3>
            <p id="alert-message"></p>
            <button onclick="closeAlert()">OK</button>
        </div>
    </div>

    <!-- SUPPORT BUTTON -->
    <?php if (!empty($settings['support_link'])): ?>
    <a href="<?php echo htmlspecialchars($settings['support_link']); ?>" target="_blank" class="floating-support-btn">
        <i class="fas fa-headset"></i>
    </a>
    <?php endif; ?>

    <script src="js/app.js"></script>
    <script>
        // Global variables
        let currentBetType = '';
        let selectedPaymentMethod = 'bKash';
        let currentBalance = <?php echo $total_balance; ?>;
        
        // Navigation
        function navigateToPage(page) {
            document.querySelectorAll('.page-content').forEach(el => el.classList.remove('active'));
            document.getElementById(`${page}-page`).classList.add('active');
            
            const mainPages = ['home', 'wallet', 'profile'];
            document.querySelector('.nav-tabs').style.display = mainPages.includes(page) ? 'flex' : 'none';
            
            // Update active tab
            document.querySelectorAll('.nav-tab').forEach(tab => tab.classList.remove('active'));
            if (mainPages.includes(page)) {
                const tab = document.querySelector(`.nav-tab[data-page="${page}"]`);
                if (tab) tab.classList.add('active');
            }
        }
        
        // Tab buttons
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.addEventListener('click', () => navigateToPage(tab.dataset.page));
        });
        
        // Betting buttons
        document.querySelectorAll('.betting-area .btn').forEach(btn => {
            btn.addEventListener('click', () => {
                if (!<?php echo $user['has_deposited'] ? 'true' : 'false'; ?>) {
                    showAlert('বেটিং করতে প্রথমে ডিপোজিট করুন', 'প্রয়োজনীয়', 'error');
                    return;
                }
                
                currentBetType = btn.dataset.betType;
                document.querySelector('#bet-popup-title span').textContent = currentBetType;
                document.getElementById('bet-amount').value = '';
                document.getElementById('bet-popup').style.display = 'flex';
            });
        });
        
        // Bet functions
        function addBetAmount(amount) {
            const input = document.getElementById('bet-amount');
            input.value = (parseFloat(input.value) || 0) + amount;
        }
        
        function multiplyBetAmount(multiplier) {
            const input = document.getElementById('bet-amount');
            input.value = (parseFloat(input.value) || 1) * multiplier;
        }
        
        function closeBetPopup() {
            document.getElementById('bet-popup').style.display = 'none';
        }
        
        function confirmBet() {
            const amount = parseFloat(document.getElementById('bet-amount').value);
            if (!amount || amount < 1) {
                showAlert('সঠিক পরিমাণ লিখুন', 'ত্রুটি', 'error');
                return;
            }
            
            if (amount > currentBalance) {
                showAlert('পর্যাপ্ত ব্যালেন্স নেই', 'ত্রুটি', 'error');
                return;
            }
            
            // Place bet via AJAX
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `ajax=1&action=place_bet&bet_type=${currentBetType}&amount=${amount}&period=${Date.now()}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showAlert(data.message, 'সফল', 'success');
                    closeBetPopup();
                    location.reload();
                } else {
                    showAlert(data.message, 'ত্রুটি', 'error');
                }
            });
        }
        
        // Deposit functions
        function switchPaymentMethod(method) {
            selectedPaymentMethod = method;
            document.querySelectorAll('.payment-tab').forEach(tab => tab.classList.remove('active'));
            document.querySelectorAll('.payment-tab').forEach(tab => {
                if (tab.textContent.includes(method)) tab.classList.add('active');
            });
            
            // Show numbers for this method
            document.querySelectorAll('.number-card').forEach(card => {
                card.style.display = card.dataset.type === method ? 'block' : 'none';
                card.classList.remove('selected');
            });
            document.querySelector(`.number-card[data-type="${method}"]`)?.classList.add('selected');
        }
        
        function setAmount(amount) {
            document.getElementById('deposit-amount-input').value = amount;
        }
        
        function submitDeposit() {
            const amount = document.getElementById('deposit-amount-input').value;
            const selectedCard = document.querySelector('.number-card.selected');
            
            if (!amount || amount < <?php echo $settings['min_deposit'] ?? MIN_DEPOSIT; ?>) {
                showAlert('সর্বনিম্ন <?php echo $settings['min_deposit'] ?? MIN_DEPOSIT; ?> টাকা ডিপোজিট করুন', 'ত্রুটি', 'error');
                return;
            }
            
            if (!selectedCard) {
                showAlert('একটি পেমেন্ট নাম্বার সিলেক্ট করুন', 'ত্রুটি', 'error');
                return;
            }
            
            const agentNumber = selectedCard.querySelector('.number').textContent;
            const transactionId = prompt('লেনদেন আইডি (TxnID) লিখুন:');
            
            if (!transactionId) {
                showAlert('লেনদেন আইডি প্রয়োজন', 'ত্রুটি', 'error');
                return;
            }
            
            // Submit deposit
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `ajax=1&action=deposit_request&amount=${amount}&method=${selectedPaymentMethod}&agent_number=${agentNumber}&transaction_id=${transactionId}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showAlert(data.message, 'সফল', 'success');
                    navigateToPage('wallet');
                } else {
                    showAlert(data.message, 'ত্রুটি', 'error');
                }
            });
        }
        
        // Withdrawal functions
        function submitWithdrawal() {
            const amount = document.getElementById('withdraw-amount').value;
            const method = document.getElementById('withdraw-method').value;
            const account = document.getElementById('withdraw-account').value;
            
            if (!amount || amount < <?php echo $settings['min_withdraw'] ?? MIN_WITHDRAW; ?>) {
                showAlert('সর্বনিম্ন <?php echo $settings['min_withdraw'] ?? MIN_WITHDRAW; ?> টাকা উত্তোলন করুন', 'ত্রুটি', 'error');
                return;
            }
            
            if (!account) {
                showAlert('অ্যাকাউন্ট নাম্বার প্রয়োজন', 'ত্রুটি', 'error');
                return;
            }
            
            if (amount > <?php echo $winning_balance; ?>) {
                showAlert('উইনিং ব্যালেন্সে পর্যাপ্ত টাকা নেই', 'ত্রুটি', 'error');
                return;
            }
            
            // Submit withdrawal
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `ajax=1&action=withdraw_request&amount=${amount}&method=${method}&account_number=${account}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showAlert(data.message, 'সফল', 'success');
                    navigateToPage('wallet');
                } else {
                    showAlert(data.message, 'ত্রুটি', 'error');
                }
            });
        }
        
        // Gift code functions
        function redeemGiftCode() {
            const code = document.getElementById('gift-code-input-field').value;
            
            if (!code) {
                showAlert('গিফট কোড লিখুন', 'ত্রুটি', 'error');
                return;
            }
            
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `ajax=1&action=redeem_gift_code&code=${code}`
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showAlert(data.message, 'সফল', 'success');
                    document.getElementById('gift-code-input-field').value = '';
                    location.reload();
                } else {
                    showAlert(data.message, 'ত্রুটি', 'error');
                }
            });
        }
        
        // Utility functions
        function copyReferralCode() {
            const code = document.getElementById('referral-code-display').textContent;
            navigator.clipboard.writeText(code).then(() => {
                showAlert('রেফারেল কোড কপি করা হয়েছে', 'সফল', 'success');
            });
        }
        
        function refreshBalance() {
            location.reload();
        }
        
        function logout() {
            window.location.href = 'logout.php';
        }
        
        function changePassword() {
            const current = document.getElementById('current-password').value;
            const newPass = document.getElementById('new-password').value;
            const confirm = document.getElementById('confirm-new-password').value;
            
            if (!current || !newPass) {
                showAlert('সব তথ্য পূরণ করুন', 'ত্রুটি', 'error');
                return;
            }
            
            if (newPass !== confirm) {
                showAlert('পাসওয়ার্ড মেলেনি', 'ত্রুটি', 'error');
                return;
            }
            
            if (newPass.length < 6) {
                showAlert('পাসওয়ার্ড কমপক্ষে ৬ অক্ষরের হতে হবে', 'ত্রুটি', 'error');
                return;
            }
            
            // Update password via AJAX
            fetch('api/update_password.php', {
                method: 'POST',
                body: JSON.stringify({
                    current_password: current,
                    new_password: newPass
                })
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showAlert('পাসওয়ার্ড পরিবর্তন সফল', 'সফল', 'success');
                    document.getElementById('current-password').value = '';
                    document.getElementById('new-password').value = '';
                    document.getElementById('confirm-new-password').value = '';
                } else {
                    showAlert(data.message, 'ত্রুটি', 'error');
                }
            });
        }
        
        function showAlert(message, title = 'Alert', type = 'info') {
            const alert = document.getElementById('alert-popup');
            document.getElementById('alert-title').textContent = title;
            document.getElementById('alert-message').textContent = message;
            alert.style.display = 'flex';
        }
        
        function closeAlert() {
            document.getElementById('alert-popup').style.display = 'none';
        }
        
        // Timer function (simplified)
        let seconds = 59;
        const countdownElement = document.getElementById('countdown');
        
        function updateTimer() {
            seconds--;
            if (seconds < 0) seconds = 59;
            
            countdownElement.textContent = `00:${seconds.toString().padStart(2, '0')}`;
            
            // Disable betting in last 5 seconds
            const disableBetting = seconds <= 5;
            document.querySelectorAll('.betting-area .btn').forEach(btn => {
                btn.disabled = disableBetting || !<?php echo $user['has_deposited'] ? 'true' : 'false'; ?>;
            });
            
            // Show overlay in last 5 seconds
            document.getElementById('timer-overlay').style.display = disableBetting ? 'flex' : 'none';
            document.getElementById('overlay-countdown-text').textContent = seconds;
        }
        
        setInterval(updateTimer, 1000);
        
        // Initialize
        switchPaymentMethod('bKash');
        
        // Tab switching for history
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.addEventListener('click', () => {
                document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                
                document.querySelectorAll('.tab-panel').forEach(panel => panel.classList.remove('active'));
                document.getElementById(`${btn.dataset.tab}-panel`).classList.add('active');
            });
        });
    </script>
</body>
</html>